package be.ehealth.businessconnector.hubv3.session;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.util.List;

import org.joda.time.DateTime;
import org.junit.ClassRule;
import org.junit.FixMethodOrder;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TestRule;
import org.junit.runners.MethodSorters;

import be.ehealth.business.common.domain.Patient;
import be.ehealth.business.intrahubcommons.exception.IntraHubBusinessConnectorException;
import be.ehealth.businessconnector.hubv3.helpers.HubConfigCommon;
import be.ehealth.businessconnector.hubv3.helpers.HubConfigVitalink;
import be.ehealth.businessconnector.hubv3.helpers.HubTestHelper;
import be.ehealth.businessconnector.testcommons.BusinessSessionRule;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.fgov.ehealth.hubservices.core.v3.AccessRightType;
import be.fgov.ehealth.hubservices.core.v3.ConsentHCPartyType;
import be.fgov.ehealth.hubservices.core.v3.ConsentType;
import be.fgov.ehealth.hubservices.core.v3.FolderType;
import be.fgov.ehealth.hubservices.core.v3.GetLatestUpdateResponse;
import be.fgov.ehealth.hubservices.core.v3.GetPatientAuditTrailResponse;
import be.fgov.ehealth.hubservices.core.v3.GetTransactionListResponse;
import be.fgov.ehealth.hubservices.core.v3.GetTransactionResponse;
import be.fgov.ehealth.hubservices.core.v3.GetTransactionSetResponse;
import be.fgov.ehealth.hubservices.core.v3.HCPartyIdType;
import be.fgov.ehealth.hubservices.core.v3.HcpartyType;
import be.fgov.ehealth.hubservices.core.v3.KmehrHeaderDeclareTransaction;
import be.fgov.ehealth.hubservices.core.v3.LocalSearchType;
import be.fgov.ehealth.hubservices.core.v3.Paginationrequestinfo;
import be.fgov.ehealth.hubservices.core.v3.PatientIdType;
import be.fgov.ehealth.hubservices.core.v3.PutTransactionResponse;
import be.fgov.ehealth.hubservices.core.v3.PutTransactionSetResponse;
import be.fgov.ehealth.hubservices.core.v3.RevokeTransactionResponse;
import be.fgov.ehealth.hubservices.core.v3.SelectGetPatientAuditTrailType;
import be.fgov.ehealth.hubservices.core.v3.SelectRevokeAccessRightType;
import be.fgov.ehealth.hubservices.core.v3.TherapeuticLinkType;
import be.fgov.ehealth.hubservices.core.v3.TransactionIdType;
import be.fgov.ehealth.hubservices.core.v3.TransactionSummaryType;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDACCESSRIGHT;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDACCESSRIGHTschemes;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDACCESSRIGHTvalues;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDTRANSACTION;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDTRANSACTIONschemes;
import be.fgov.ehealth.standards.kmehr.id.v1.IDKMEHR;
import be.fgov.ehealth.standards.kmehr.schema.v1.AuthorType;
import be.fgov.ehealth.standards.kmehr.schema.v1.ErrorType;
import be.fgov.ehealth.standards.kmehr.schema.v1.Kmehrmessage;
import be.fgov.ehealth.standards.kmehr.schema.v1.PersonType;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;

@FixMethodOrder(MethodSorters.NAME_ASCENDING)
public class WalkThroughVitalinkIntegrationTest {

    @ClassRule
    public static TestRule sessionPrerequisite = BusinessSessionRule.withActiveSession("/be.ehealth.businessconnector.intrahubv3.test.properties");

    @ClassRule
    public static InitHubRule hubPrerequesite = new InitVitalinkHubRule();

    @Rule
    public ExpectedException expectedException = ExpectedException.none();

    /**
     * SL for medication scheme transaction identification (platform-dependant)
     */
    private static final String TRANSACTION_URI = "vitalinkuri";

    private static String transactionId;

    private static int latestMedicationSchemeVersion;

    private final HubService hubService = hubPrerequesite.getHubService();

    private HubTestHelper helper = new HubTestHelper("vitalink", new Patient.Builder().withFirstName(HubConfigVitalink.PATIENT_FIRSTNAME).withFamilyName(HubConfigVitalink.PATIENT_LASTNAME).withInss(HubConfigVitalink.PATIENT_NISS).build());


    @Test
    public void _010_putHcPartyConsent() throws Exception {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");

        ConsentHCPartyType consent = new ConsentHCPartyType();
        consent.setAuthor(helper.createAuthor());
        consent.setHcparty(helper.createHcPartyIdProfessional());
        consent.setRevokedate(new DateTime());
        consent.setSigndate(new DateTime());
        hubService.putHCPartyConsent(consent);
    }

    @Test
    public void _020_putPatient() throws Exception {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");
        PersonType person = helper.createPatient();
        hubService.putPatient(person);
    }

    @Test
    public void _030_getPatient() throws IntraHubBusinessConnectorException, TechnicalConnectorException {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");

        hubService.getPatient(helper.createPatientIdType());
    }

    @Test
    public void _040_putPatientConsent() throws Exception {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");

        hubService.putPatientConsent(helper.createConsentType());
    }

    @Test
    public void _050_putTherapeuticLink() throws Exception {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");

        TherapeuticLinkType therapeuticLink = helper.createTherapeuticLinkType();
        hubService.putTherapeuticLink(therapeuticLink);
    }

    @Test
    public void _060_putTransaction() throws Exception {
        Kmehrmessage kmehrmessage = helper.createPutTransactionMessage();

        PutTransactionResponse putTransaction = hubService.putTransaction(kmehrmessage);
        transactionId = putTransaction.getTransaction().getIds().get(0).getValue();

        XmlAsserter.assertSimilar(helper.expectedResponse("PutTransaction"), ConnectorXmlUtils.toString(putTransaction));
    }

    @Test
    public void _070_getPatientConsent() throws TechnicalConnectorException, IntraHubBusinessConnectorException {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");
        hubService.getPatientConsent(helper.createSelectGetPatientConsentType());
    }

    @Test
    public void _080_getTherapeuticLink() throws Exception {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");

        hubService.getTherapeuticLink(helper.createSelectGetHCPartyPatientConsentType());
    }

    @Test
    public void _090_getTransactionList() throws Exception {        
        PatientIdType patientId = helper.createPatientIdType();
        GetTransactionListResponse transactionListResponse = hubService.getTransactionList(patientId, LocalSearchType.GLOBAL, helper.createTransactionWithPeriodType());

        List<ErrorType> errors = transactionListResponse.getAcknowledge().getErrors();
        assertTrue("There should be one error element", errors.size() == 1);
        assertEquals("The status code should be 200(success)", "200", errors.get(0).getCds().get(0).getValue());
        assertTrue("Transaction list should not be empty", transactionListResponse.getKmehrheader().getFolder().getTransactions().size() > 0);
    }

    @Test
    public void _091_getTransactionList_pagination() throws Exception {
        GetTransactionListResponse transactionListResponse = testGetTransactionListPagination(1);
        
        List<ErrorType> errors = transactionListResponse.getAcknowledge().getErrors();
        assertTrue("There should be one error element", errors.size() == 1);
        assertEquals("The status code should be 200(success)", "200", errors.get(0).getCds().get(0).getValue());
        assertTrue("Transaction list should not be empty", transactionListResponse.getKmehrheader().getFolder().getTransactions().size() > 0);
    }

    @Test
    public void _091_getTransactionList_pagination_index0() throws Exception {
        GetTransactionListResponse transactionListResponse = testGetTransactionListPagination(0);
        
        List<ErrorType> errors = transactionListResponse.getAcknowledge().getErrors();
        assertTrue("There should be one error element", errors.size() == 1);
        assertEquals("The status code should be 494", "494", errors.get(0).getCds().get(0).getValue());
    }

    @Test
    public void _091_getTransactionList_pagination_index10() throws Exception {
        GetTransactionListResponse transactionListResponse = testGetTransactionListPagination(10);

        List<ErrorType> errors = transactionListResponse.getAcknowledge().getErrors();
        assertTrue("There should be one error element", errors.size() == 1);
        assertEquals("The status code should be 495", "495", errors.get(0).getCds().get(0).getValue());
    }

    private GetTransactionListResponse testGetTransactionListPagination(int index) throws TechnicalConnectorException, IntraHubBusinessConnectorException {
        PatientIdType patientId = helper.createPatientIdType();
        Paginationrequestinfo pagReqInfo = new Paginationrequestinfo();
        pagReqInfo.setIndex(index);
        return hubService.getTransactionList(patientId, LocalSearchType.GLOBAL, helper.createTransactionWithPeriodType(), pagReqInfo);
    }

    @Test
    public void _100_getTransaction() throws Exception {
        PatientIdType patientId = helper.createPatientIdType();
        GetTransactionResponse transactionResponse = hubService.getTransaction(patientId, helper.createTransactionBaseType(TRANSACTION_URI, transactionId));

        XmlAsserter.assertSimilar(helper.expectedResponse("GetTransaction"), ConnectorXmlUtils.toString(transactionResponse));
    }

    @Test
    public void _110_revokeTransaction() throws Exception {
        RevokeTransactionResponse revokeTransaction = hubService.revokeTransaction(helper.createPatientIdType(), helper.createTransactionBaseType(TRANSACTION_URI, transactionId));

        String expected = helper.expectedResponse("RevokeTransaction");
        XmlAsserter.assertSimilar(expected, revokeTransaction);
    }

    @Test
    public void _120_getLastestUpdate() throws Exception {
        GetLatestUpdateResponse latestUpdateResponse = hubService.getLatestUpdate(helper.createSelectGetLatestUpdateType());
        latestMedicationSchemeVersion = helper.getLatestMedicationSchemeVersion(latestUpdateResponse);

        XmlAsserter.assertSimilar(helper.expectedResponse("GetLatestUpdate"), ConnectorXmlUtils.toString(latestUpdateResponse));
    }

    @Test
    public void _130_putTransactionSet() throws Exception {
        Kmehrmessage kmehrmessage = helper.createTransactionSetMessage(latestMedicationSchemeVersion);

        PutTransactionSetResponse putTransactionSet = hubService.putTransactionSet(kmehrmessage);

        XmlAsserter.assertSimilar(helper.expectedResponse("PutTransactionSet"), ConnectorXmlUtils.toString(putTransactionSet));
    }

    @Test
    public void _140_getTransactionSet() throws Exception {
        PatientIdType patientId = helper.createPatientIdType();
        StringBuilder transactionSetIdBuilder = new StringBuilder().append("/subject/").append(HubConfigVitalink.PATIENT_NISS).append("/medication-scheme");

        GetTransactionSetResponse transactionResponse = hubService.getTransactionSet(patientId, helper.createTransactionBaseType(TRANSACTION_URI, transactionSetIdBuilder.toString()));

        XmlAsserter.assertSimilar(helper.expectedResponse("GetTransactionSet"), ConnectorXmlUtils.toString(transactionResponse));
    }

    @Test
    public void _150_getPatientAuditTrail() throws Exception {
        SelectGetPatientAuditTrailType patientAudiTrail = new SelectGetPatientAuditTrailType();
        patientAudiTrail.setBegindate(new DateTime());
        patientAudiTrail.setEnddate(new DateTime());
        patientAudiTrail.setPatient(helper.createPatientIdType());
        patientAudiTrail.setTransaction(helper.createTransactionBaseType(TRANSACTION_URI, transactionId));
        GetPatientAuditTrailResponse getPatientAuditTrailResponse = hubService.getPatientAuditTrail(patientAudiTrail);

        assertEquals("Role [physician] should not be allowed to perform action", "367", getPatientAuditTrailResponse.getAcknowledge().getErrors().get(0).getCds().get(0).getValue());
    }

    @Test
    public void _151_getPatientAuditTrail_pagination() throws Exception {
        Paginationrequestinfo pagReqInfo = new Paginationrequestinfo();
        pagReqInfo.setIndex(1);

        SelectGetPatientAuditTrailType patientAudiTrail = new SelectGetPatientAuditTrailType();
        patientAudiTrail.setBegindate(new DateTime());
        patientAudiTrail.setEnddate(new DateTime());
        patientAudiTrail.setPatient(helper.createPatientIdType());
        patientAudiTrail.setTransaction(helper.createTransactionBaseType(TRANSACTION_URI, transactionId));
        GetPatientAuditTrailResponse getPatientAuditTrailResponse = hubService.getPatientAuditTrail(patientAudiTrail, pagReqInfo);

        assertEquals("Role [physician] should not be allowed to perform action", "367", getPatientAuditTrailResponse.getAcknowledge().getErrors().get(0).getCds().get(0).getValue());
    }

    @Test
    public void _160_declareTransaction() throws Exception {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");

        KmehrHeaderDeclareTransaction kmehrHeader = new KmehrHeaderDeclareTransaction();
        FolderType folder = new FolderType();
        PatientIdType patient = helper.createPatientIdType();

        TransactionSummaryType transaction = new TransactionSummaryType();
        AuthorType author = helper.createAuthor();
        transaction.setAuthor(author);
        CDTRANSACTION cdTransaction = new CDTRANSACTION();
        cdTransaction.setS(CDTRANSACTIONschemes.CD_TRANSACTION);
        cdTransaction.setSV("1.0");
        cdTransaction.setValue("sumehr");
        transaction.getCds().add(cdTransaction);
        transaction.setDate(new org.joda.time.DateTime());
        transaction.getIds().add(helper.createMessageId(HubConfigCommon.MESSAGE_ID));
        transaction.setIscomplete(true);
        transaction.setIsvalidated(true);
        transaction.setTime(new org.joda.time.DateTime());
        transaction.setRecorddatetime(new org.joda.time.DateTime());

        folder.setTransaction(transaction);
        folder.setPatient(patient);
        kmehrHeader.setFolder(folder);
        hubService.declareTransaction(kmehrHeader);
    }

    @Test
    public void _170_requestPublication() throws IntraHubBusinessConnectorException, TechnicalConnectorException {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");

        hubService.requestPublication(helper.createPatientIdType(), helper.createTransactionWithPeriodType(), "");
    }

    @Test
    public void _180_putHcParty() throws TechnicalConnectorException, IntraHubBusinessConnectorException {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");

        hubService.putHCParty(helper.createHcPartyAdaptedType());
    }

    @Test
    public void _190_getHcParty() throws TechnicalConnectorException, IntraHubBusinessConnectorException {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");

        HCPartyIdType hcParty = helper.createHcPartyIdProfessional();
        hubService.getHCParty(hcParty);
    }

    @Test
    public void _200_getHcPartyConsent() throws TechnicalConnectorException, IntraHubBusinessConnectorException {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");

        HCPartyIdType hcParty = helper.createHcPartyIdProfessional();
        hubService.getHCPartyConsent(hcParty);
    }

    @Test
    public void _210_revokeHcPartyConsent() throws TechnicalConnectorException, IntraHubBusinessConnectorException {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");

        ConsentHCPartyType consent = new ConsentHCPartyType();
        consent.setAuthor(helper.createAuthor());
        consent.setHcparty(helper.createHcPartyIdProfessional());
        consent.setRevokedate(new DateTime());
        consent.setSigndate(new DateTime());

        hubService.revokeHCPartyConsent(consent);
    }

    @Test
    public void _220_revokeTherapeuticLink() throws Exception {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");

        TherapeuticLinkType therapeuticLink = helper.createTherapeuticLinkType();
        therapeuticLink.setEnddate(new DateTime());
        hubService.revokeTherapeuticLink(therapeuticLink);
    }

    @Test
    public void _230_revokePatientConsent() throws Exception {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");

        ConsentType consent = helper.createConsentType();
        consent.setRevokedate(new DateTime());
        hubService.revokePatientConsent(consent);
    }

    @Test
    public void _240_putAccessRight() throws TechnicalConnectorException, IntraHubBusinessConnectorException {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");

        AccessRightType accessRight = new AccessRightType();
        CDACCESSRIGHT cdAccessRight = new CDACCESSRIGHT();
        cdAccessRight.setS(CDACCESSRIGHTschemes.CD_ACCESSRIGHT);
        cdAccessRight.setSV("1.0");
        cdAccessRight.setValue(CDACCESSRIGHTvalues.ALLOW);
        accessRight.setCd(cdAccessRight);
        be.fgov.ehealth.hubservices.core.v3.HcpartyType hcParty = new be.fgov.ehealth.hubservices.core.v3.HcpartyType();
        hcParty.getCds().add(helper.createCdHcPartyProfession());
        hcParty.getIds().add(helper.createIdHcPartyNihii());
        accessRight.setHcparty(hcParty);
        TransactionIdType transaction = helper.createTransactionIdType(transactionId);
        accessRight.setTransaction(transaction);
        hubService.putAccessRight(accessRight);
    }

    @Test
    public void _250_getAccessRight() throws IntraHubBusinessConnectorException, TechnicalConnectorException {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");

        TransactionIdType transaction = new TransactionIdType();
        IDKMEHR id = helper.createMessageId(HubConfigCommon.MESSAGE_ID);
        transaction.getIds().add(id);
        hubService.getAccessRight(transaction);
    }

    @Test
    public void _260_revokeAccessRight() throws IntraHubBusinessConnectorException, TechnicalConnectorException {
        expectedException.expect(TechnicalConnectorException.class);
        expectedException.expectMessage("Unknown error occured : Not supported");

        SelectRevokeAccessRightType accessRight = new SelectRevokeAccessRightType();
        HcpartyType hcParty = new HcpartyType();
        hcParty.getCds().add(helper.createCdHcPartyProfession());
        hcParty.getIds().add(helper.createIdHcPartyNihii());
        accessRight.setHcparty(hcParty);
        accessRight.setTransaction(helper.createTransactionIdType(transactionId));
        hubService.revokeAccessRight(accessRight);
    }
}
