/*
 * Copyright (c) eHealth
 */
package be.ehealth.business.mycarenetcommons.builders.impl;

import be.ehealth.business.mycarenetcommons.builders.ResponseBuilder;
import be.ehealth.business.mycarenetcommons.builders.ResponseBuilderFactory;
import be.ehealth.business.mycarenetcommons.builders.util.BlobUtil;
import be.ehealth.business.mycarenetdomaincommons.exception.ConnectorValidationException;
import be.ehealth.business.mycarenetdomaincommons.exception.InvalidBlobContentConnectorException;
import be.ehealth.business.mycarenetcommons.mapper.SendRequestMapper;
import be.ehealth.business.mycarenetdomaincommons.builders.BlobBuilder;
import be.ehealth.business.mycarenetdomaincommons.builders.BlobBuilderFactory;
import be.ehealth.business.mycarenetdomaincommons.builders.impl.BuilderUtils;
import be.ehealth.business.mycarenetdomaincommons.domain.Blob;
import be.ehealth.technicalconnector.config.ConfigFactory;
import be.ehealth.technicalconnector.config.ConfigValidator;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.fgov.ehealth.mycarenet.commons.core.v2.BlobType;
import be.fgov.ehealth.mycarenet.commons.core.v2.CommonOutputType;
import be.fgov.ehealth.mycarenet.commons.protocol.v2.ResponseReturnType;
import be.fgov.ehealth.mycarenet.commons.protocol.v2.SendResponseType;
import be.fgov.ehealth.technicalconnector.tests.session.SessionDestroyer;
import be.fgov.ehealth.technicalconnector.tests.session.SessionInitializer;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;
import org.w3._2005._05.xmlmime.Base64Binary;

import java.security.NoSuchAlgorithmException;


/**
 * test class to test the factory.
 * 
 * @author EH058
 * 
 */
public class ResponseBuilderIntegrationTest {

    /**
     * @throws java.lang.Exception
     */
    @BeforeClass
    public static void setUpSession() throws Exception {
        SessionInitializer.init("/be.ehealth.businessconnector.mycarenetcommons.test.properties");
        // to be able to test with deflate , you also need to load the following module : ConfigurationModuleRegisterTransformers
        addConfigModuleAndReload("be.fgov.ehealth.technicalconnector.signature.config.impl.ConfigurationModuleRegisterTransformers");
    }

    /**
     * @param string
     */
    private static void addConfigModuleAndReload(String moduleName) throws TechnicalConnectorException {
        ConfigValidator configValidator = ConfigFactory.getConfigValidator();
        int i = 1;
        String nextModuleProperty = configValidator.getProperty("connector.configmodule." + i);
        while (nextModuleProperty != null && !nextModuleProperty.isEmpty()) {
            i++;
            nextModuleProperty = configValidator.getProperty("connector.configmodule." + i);
        }
        configValidator.setProperty("connector.configmodule." + i, moduleName);
        ConfigFactory.getConfigValidator().reload();

    }

    @AfterClass
    public static void destroySession() throws Exception {
        SessionDestroyer.destroy();
    }

    /**
     * Test method for {@link be.ehealth.business.mycarenetcommons.builders.impl.ResponseBuilderFactory#getResponseBuilder()}.
     */
    @SuppressWarnings("deprecation")
    @Test
    public void testGetResponseBuilderFactory() throws Exception {
        Assert.assertNotNull(ResponseBuilderFactory.getResponseBuilder());
        Assert.assertNotNull(ResponseBuilderFactory.getResponseBuilder("default"));
    }

    /**
     * Test method for {@link be.ehealth.business.mycarenetcommons.builders.impl.ResponseBuilderFactory#getResponseBuilder()}.
     */
    @Test
    public void testRightResponse() throws Exception {
        ResponseBuilder responseBuilder = ResponseBuilderFactory.getResponseBuilder("default");
        SendResponseType responseType = initResponse();
        Assert.assertEquals("<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?><blabla><BoxId><Id>10013368001</Id></BoxId></blabla>", responseBuilder.getResponse(responseType, true));
    }

    /**
     * @return
     */
    private SendResponseType initResponse() throws TechnicalConnectorException, NoSuchAlgorithmException {
        SendResponseType responseReturnType = createResponseFilledResponseReturnType();
        Base64Binary generateXades = BlobUtil.generateXades(responseReturnType.getReturn().getDetail());
        responseReturnType.getReturn().setXadesT(generateXades);
        CommonOutputType commonOutput = new CommonOutputType();
        commonOutput.setInputReference("test");
        commonOutput.setNIPReference("test");
        commonOutput.setOutputReference("test");
        responseReturnType.getReturn().setCommonOutput(commonOutput);

        return responseReturnType;
    }

    /**
     * @return
     * @throws TechnicalConnectorException
     * @throws InvalidBlobContentConnectorException
     * @throws NoSuchAlgorithmException
     */
    public SendResponseType createResponseFilledResponseReturnType() throws TechnicalConnectorException, InvalidBlobContentConnectorException, NoSuchAlgorithmException {
        BlobBuilder builder = BlobBuilderFactory.getBlobBuilder("default");
        Blob blob = builder.build("<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?><blabla><BoxId><Id>10013368001</Id></BoxId></blabla>".getBytes(), "deflate", "blob", "text/xml");
        byte[] decompressBlob = BuilderUtils.decompressBlob(blob.getContent(), blob.getContentEncoding());
        blob.setHashValue(BuilderUtils.buildHash(decompressBlob));
        BlobType blobType = SendRequestMapper.mapBlobToBlobType(blob);
        SendResponseType responseType = new SendResponseType();

        ResponseReturnType responseReturnType = new ResponseReturnType();
        responseReturnType.setDetail(blobType);
        responseType.setReturn(responseReturnType);


        return responseType;
    }

    /**
     * Test method for {@link be.ehealth.business.mycarenetcommons.builders.impl.ResponseBuilderFactory#getResponseBuilder()}.
     */
    @Test
    public void testWrongHash() throws Exception {
        ResponseBuilder responseBuilder = ResponseBuilderFactory.getResponseBuilder("default");
        SendResponseType responseType = initResponse();
        responseType.getReturn().getDetail().setHashValue("wrong".getBytes());
        try {
            responseBuilder.getResponse(responseType, true);
            Assert.fail("should have thrown exception because hash value was not correct");
        } catch (InvalidBlobContentConnectorException e) {
            Assert.assertEquals("Hash value is different : was d3Jvbmc= but expecting 8Xce5nEDVODLUoxnqkB1UwoHSlKTW5S4KttSg94snuA=", e.getMessage());
        }
    }

    /**
     * Test method for {@link be.ehealth.business.mycarenetcommons.builders.impl.ResponseBuilderFactory#getResponseBuilder()}.
     */
    @Ignore
    public void testWrongXades() throws Exception {
        ResponseBuilder responseBuilder = ResponseBuilderFactory.getResponseBuilder("default");
        SendResponseType responseType = initResponse();
        Base64Binary value = new Base64Binary();
        value.setValue("wrong".getBytes());
        responseType.getReturn().setXadesT(value);
        responseBuilder.getResponse(responseType, true);
    }

    @Test
    public void testValidateXadesWithConfigPropertyMissingAndNoXades() throws Exception {
        try {
            SendResponseType responseReturnType = createResponseFilledResponseReturnType();
            ResponseBuilder responseBuilder = ResponseBuilderFactory.getResponseBuilder("unconfiguredProject");
            responseBuilder.validateXades(responseReturnType);
            Assert.fail("a default check on level xades will be done if the property is not configured");
        } catch (InvalidBlobContentConnectorException e) {
            Assert.assertEquals("The xades value is required but null", e.getMessage());
        }
    }

    @Test
    public void testValidateXadesWithNoXadesPresentAndNoneNeeded() throws Exception {
        SendResponseType responseReturnType = createResponseFilledResponseReturnType();
        ConfigFactory.getConfigValidator().setProperty("mycarenet.noXadesNeededProject.response.neededxadeslevel", "none");
        ResponseBuilder responseBuilder = ResponseBuilderFactory.getResponseBuilder("noXadesNeededProject");
        responseBuilder.validateXades(responseReturnType);

    }

    @Test
    public void testValidateXadesWithNoXadesPresentAndNeeded() throws Exception {
        try {
            SendResponseType responseReturnType = createResponseFilledResponseReturnType();
            ConfigFactory.getConfigValidator().setProperty("mycarenet.xadesBesNeededProject.response.neededxadeslevel", "xades");
            ResponseBuilder responseBuilder = ResponseBuilderFactory.getResponseBuilder("xadesBesNeededProject");
            responseBuilder.validateXades(responseReturnType);
            Assert.fail("should have thrown InvalidBlobContentConnectorException");
        } catch (InvalidBlobContentConnectorException e) {
            Assert.assertEquals("The xades value is required but null", e.getMessage());
        }
    }

    @Test
    public void testValidateXadesWithXadesBesPresentAndXadesTNeeded() throws Exception {
        try {
            SendResponseType responseReturnType = createResponseFilledResponseReturnType();
            ConfigFactory.getConfigValidator().setProperty("mycarenet.xadesTNeededProject.response.neededxadeslevel", "xadest");
            ResponseBuilder responseBuilder = ResponseBuilderFactory.getResponseBuilder("xadesTNeededProject");
            // add xadesBes
            ConfigFactory.getConfigValidator().setProperty("mycarenet.xadesGenerationProject.request.xadestype", "xades");
            responseReturnType.getReturn().setXadesT(BlobUtil.generateXades(responseReturnType.getReturn().getDetail(), "xadesGenerationProject"));
            responseBuilder.validateXades(responseReturnType);
            Assert.fail("a xades message should not pass if xadest is required");
        } catch (InvalidBlobContentConnectorException e) {
            Assert.assertTrue(e.getMessage() != null && e.getMessage().indexOf("Xades validation failed with the following error :$error: XADES_ENCAPSULATED_TIMESTAMP_NOT_FOUND:[Error indication that there was no encapsulated timestamp in the xades-t.]") != -1);
        }
    }

    @Test
    public void testValidateXadesWithXadesTPresentAndXadesTNeeded() throws Exception {
        SendResponseType responseReturnType = createResponseFilledResponseReturnType();
        ConfigFactory.getConfigValidator().setProperty("mycarenet.xadesTNeededProject.response.neededxadeslevel", "xadest");
        ResponseBuilder responseBuilder = ResponseBuilderFactory.getResponseBuilder("xadesTNeededProject");
        // add xadesT
        ConfigFactory.getConfigValidator().setProperty("mycarenet.xadesTGenerationProject.request.xadestype", "xadest");
        responseReturnType.getReturn().setXadesT(BlobUtil.generateXades(responseReturnType.getReturn().getDetail(), "xadesTGenerationProject"));
        responseBuilder.validateXades(responseReturnType);
    }

    @Test
    public void testValidateXadesWithXadesTPresentAndXadesBesNeeded() throws Exception {
        // Xades-t = xades-bes + timestamp, so this is ok
        SendResponseType responseReturnType = createResponseFilledResponseReturnType();
        ConfigFactory.getConfigValidator().setProperty("mycarenet.xadesBesNeededProject.response.neededxadeslevel", "xades");
        ResponseBuilder responseBuilder = ResponseBuilderFactory.getResponseBuilder("xadesBesNeededProject");
        // add xadesT
        ConfigFactory.getConfigValidator().setProperty("mycarenet.xadesTGenerationProject.request.xadestype", "xadest");
        responseReturnType.getReturn().setXadesT(BlobUtil.generateXades(responseReturnType.getReturn().getDetail(), "xadesTGenerationProject"));
        responseBuilder.validateXades(responseReturnType);
    }


    @Test
    public void testWrongCommonOutput() throws Exception {
        try {
            ResponseBuilder responseBuilder = ResponseBuilderFactory.getResponseBuilder("default");
            SendResponseType responseType = initResponse();
            responseType.getReturn().getCommonOutput().setInputReference(null);
            responseBuilder.getResponse(responseType, true);
            Assert.fail("must generate a ConnectorValidationException");
        } catch (ConnectorValidationException e) {
            // success
        }
    }

    // /**
    // * @return
    // */
    // private SendResponseType initResponse() {
    // SendResponseType responseType = new SendResponseType();
    // BlobType blobType = new BlobType();
    // blobType.setContentEncoding("none");
    // blobType.setContentType("text/xml");
    // blobType.setHashValue(Base64.decode("DuZxtaOXj899T8wjQ5DJqpPu/qyw1hVL0PO7p4EQoPY="));
    // blobType.setId("blob");
    // blobType
    // .setValue("PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiIHN0YW5kYWxvbmU9InllcyI/Pgo8bnMyOnJlZ2lzdHJhdGlvbnNBbnN3ZXIgeG1sbnM9InVybjpiZTpjaW46bXljYXJlbmV0OmVzYjpjb21tb246djIiIHhtbG5zOm5zMj0idXJuOmJlOmNpbjpuaXA6c3luYzpyZWc6djEiIHhtbG5zOm5zMz0idXJuOmJlOmNpbjp0eXBlczp2MSIgeG1sbnM6eHNpPSJodHRwOi8vd3d3LnczLm9yZy8yMDAxL1hNTFNjaGVtYS1pbnN0YW5jZSIgeHNpOnNjaGVtYUxvY2F0aW9uPSJodHRwOi8vd3d3LmVoZWFsdGguZmdvdi5iZS9tZXNzYWdlc2VydmljZXMvcHJvdG9jb2wvdjEgLi4veHNkLW1lc3NhZ2VzZXJ2aWNlcy0xLTAvZWhlYWx0aC1tZXNzYWdlc2VydmljZXMvWFNEL2VoZWFsdGhfbWVzc2FnZV9wcm90b2NvbC0xXzAueHNkIj4KICAgIDxuczI6cmVnaXN0cmFudD4KICAgICAgICA8bnMyOkNhcmVQcm92aWRlcj4KICAgICAgICAgICAgPE5paGlpPgogICAgICAgICAgICAgICAgPFF1YWxpdHk+ZG9jdG9yPC9RdWFsaXR5PgogICAgICAgICAgICAgICAgPFZhbHVlPjE4MzM0NzgwMDA0PC9WYWx1ZT4KICAgICAgICAgICAgPC9OaWhpaT4KICAgICAgICA8L25zMjpDYXJlUHJvdmlkZXI+CiAgICA8L25zMjpyZWdpc3RyYW50PgogICAgPG5zMjpyZWdpc3RyYXRpb25BbnN3ZXIgc2VydmljZU5hbWU9IkdNRCIgc3RhdHVzPSJFUlJPUiI+CiAgICAgICAgPG5zMjphbnN3ZXJEZXRhaWxzPgogICAgICAgICAgICA8bnMzOkRldGFpbENvZGU+MTY4PC9uczM6RGV0YWlsQ29kZT4KICAgICAgICAgICAgPG5zMzpEZXRhaWxTb3VyY2U+NTAwPC9uczM6RGV0YWlsU291cmNlPgogICAgICAgICAgICA8bnMzOkxvY2F0aW9uPipbbG9jYWwtbmFtZSgpID0gJ3JlZ2lzdHJhdGlvbnMnIGFuZCBuYW1lc3BhY2UtdXJpKCk9J3VybjpiZTpjaW46bmlwOnN5bmM6cmVnOnYxJyBdLypbbG9jYWwtbmFtZSgpID0gJ3JlZ2lzdHJhbnQnIGFuZCBuYW1lc3BhY2UtdXJpKCk9J3VybjpiZTpjaW46bmlwOnN5bmM6cmVnOnYxJyBdLypbbG9jYWwtbmFtZSgpID0gJ0NhcmVQcm92aWRlcicgYW5kIG5hbWVzcGFjZS11cmkoKT0ndXJuOmJlOmNpbjpuaXA6c3luYzpyZWc6djEnIF08L25zMzpMb2NhdGlvbj4KICAgICAgICAgICAgPG5zMzpNZXNzYWdlIHhtbDpsYW5nPSJubCI+RG9rdGVyIGlzIHJlZWRzIGdlYWZmaWxpZWVyZCB0b3QgTXlDYXJlbmV0IFNlcnZpY2VzLjwvbnMzOk1lc3NhZ2U+CiAgICAgICAgPC9uczI6YW5zd2VyRGV0YWlscz4KICAgIDwvbnMyOnJlZ2lzdHJhdGlvbkFuc3dlcj4KPC9uczI6cmVnaXN0cmF0aW9uc0Fuc3dlcj4K".getBytes());
    // ResponseReturnType responseReturnType = new ResponseReturnType();
    // responseReturnType.setDetail(blobType);
    // Base64Binary value = new Base64Binary();
    // value
    // .setValue("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".getBytes());
    // responseReturnType.setXadesT(value);
    // CommonOutputType commonOutput = new CommonOutputType();
    // commonOutput.setInputReference("test");
    // commonOutput.setNIPReference("test");
    // commonOutput.setOutputReference("test");
    // responseReturnType.setCommonOutput(commonOutput);
    // responseType.setReturn(responseReturnType);
    // return responseType;
    // }
    // be.ehealth.business.mycarenetdomaincommons.exception.InvalidBlobContentConnectorException: Hash value is different : was
    // TUVZd01uaHZSR1V2UVRkSWNXaDVSRXhtUldWSVZsbFpXR1JtVFZWeWFsazBRVk12VVRsTVdHUnlNRDA9 but expecting
    // 0cLHGZAHVcJMH8ue34PA68R3Kx0TpNFUUmAYHlojjw4=

}
