/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.wsconsent.service;

import be.ehealth.business.kmehrcommons.CDConsentBuilderUtil;
import be.ehealth.business.kmehrcommons.helper.ErrorTypeManagement;
import be.ehealth.businessconnector.wsconsent.builders.RequestObjectBuilderFactory;
import be.ehealth.businessconnector.wsconsent.exception.WsConsentBusinessConnectorException;
import be.ehealth.businessconnector.wsconsent.session.WsConsentSessionServiceFactory;
import be.ehealth.businessconnector.wsconsent.utils.TestUtils;
import be.ehealth.technicalconnector.exception.ConnectorException;
import be.ehealth.technicalconnector.exception.SessionManagementException;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.session.Session;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.fgov.ehealth.hubservices.core.v2.*;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDCONSENT;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDCONSENTvalues;
import be.fgov.ehealth.standards.kmehr.id.v1.IDHCPARTY;
import be.fgov.ehealth.standards.kmehr.id.v1.IDHCPARTYschemes;
import be.fgov.ehealth.standards.kmehr.schema.v1.ErrorType;
import be.fgov.ehealth.standards.kmehr.schema.v1.HcpartyType;
import be.fgov.ehealth.technicalconnector.tests.session.SessionInitializer;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;
import org.joda.time.DateTime;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;


/**
 * Common class for ws-consent integration tests
 */
public abstract class AbstractWsConsentServiceIntegrationTest {

    static final Logger LOG = LoggerFactory.getLogger(AbstractWsConsentServiceIntegrationTest.class);

    @BeforeClass
    public static void init() throws Exception {
        SessionInitializer.init("/be.ehealth.businessconnector.wsconsent.test.properties");
    }

    @AfterClass
    public static void tearDown() throws Exception {
        Session.getInstance().unloadSession();
    }

    PutPatientConsentResponse putConsentForCurrentPatient(ConsentType consentType) throws TechnicalConnectorException, WsConsentBusinessConnectorException, InstantiationException, SessionManagementException {
        PutPatientConsentRequest consentRequest = RequestObjectBuilderFactory.getRequestObjectBuilder().createPutRequest(getAuthor(), consentType);
        ConnectorXmlUtils.dump(consentRequest);

        // Service
        WsConsentService service = ServiceFactory.getWsConsentService();
        PutPatientConsentResponse response = service.putPatientConsent(Session.getInstance().getSession().getSAMLToken(), consentRequest);
        return response;
    }

    static AuthorWithPatientAndPersonType getAuthor() throws TechnicalConnectorException, WsConsentBusinessConnectorException, InstantiationException {
        AuthorWithPatientAndPersonType author = RequestObjectBuilderFactory.getAuthorBuilder().createAuthor();
        return author;
    }

    PatientIdType getPatient() throws TechnicalConnectorException {
        PatientIdType patient = RequestObjectBuilderFactory.getPatientInfoBuilder().readFromEidCard();
        return patient;
    }

    GetPatientConsentResponse retrievePatientConsentResponse() throws TechnicalConnectorException, WsConsentBusinessConnectorException, InstantiationException, SessionManagementException {
        List<CDCONSENT> consentList = new ArrayList<CDCONSENT>();
        consentList.add(CDConsentBuilderUtil.createCDConsent("1.0", CDCONSENTvalues.RETROSPECTIVE));

        SelectGetPatientConsentType consentType = RequestObjectBuilderFactory.getConsentBuilder().createSelectGetPatientConsent(getPatient(), consentList);
        AuthorWithPatientAndPersonType author = getAuthor();

        GetPatientConsentRequest consentRequest = RequestObjectBuilderFactory.getRequestObjectBuilder().createGetRequest(author, consentType);

        TestUtils.logJaxbObject(consentRequest);
        // Service
        WsConsentService service = ServiceFactory.getWsConsentService();
        GetPatientConsentResponse response = service.getPatientConsent(Session.getInstance().getSession().getSAMLToken(), consentRequest);

        TestUtils.logJaxbObject(response);
        return response;
    }

    RevokePatientConsentResponse revokePatientconsent(ConsentType consentType) throws TechnicalConnectorException, WsConsentBusinessConnectorException, InstantiationException, SessionManagementException {

        consentType.setRevokedate(new DateTime());
        AuthorWithPatientAndPersonType author = getAuthor();

        RevokePatientConsentRequest consentRequest = RequestObjectBuilderFactory.getRequestObjectBuilder().createRevokeRequest(author, consentType);

        TestUtils.logJaxbObject(consentRequest);

        // Service
        WsConsentService service = ServiceFactory.getWsConsentService();
        RevokePatientConsentResponse response = service.revokePatientConsent(Session.getInstance().getSession().getSAMLToken(), consentRequest);

        TestUtils.logJaxbObject(response);

        LOG.debug("Response: " + response);
        LOG.debug("isComplete =" + response.getAcknowledge().isIscomplete());
        List<ErrorType> errors = response.getAcknowledge().getErrors();
        for (ErrorType error : errors) {
            LOG.debug("URL : " + error.getUrl() + " errorMessage : " + error.getDescription().getValue());
        }
        return response;
    }

    ConsentType createNewConsentTypeForCurrentPatientAndAuthor() throws TechnicalConnectorException, WsConsentBusinessConnectorException, InstantiationException {
        List<CDCONSENT> consentList = new ArrayList<CDCONSENT>();
        consentList.add(CDConsentBuilderUtil.createCDConsent("1.0", CDCONSENTvalues.RETROSPECTIVE));
        ConsentType consentType = RequestObjectBuilderFactory.getConsentBuilder().createNewConsent(getPatient(), consentList, new DateTime(), getAuthor());
        return consentType;
    }

    PutPatientConsentResponse putConsentForCurrentPatientWithSessionService(ConsentType consentType) throws InstantiationException, ConnectorException {
        PutPatientConsentRequest consentRequest = RequestObjectBuilderFactory.getRequestObjectBuilder().createPutRequest(getAuthor(), consentType);
        TestUtils.logJaxbObject(consentRequest);
        // Service
        be.ehealth.businessconnector.wsconsent.session.WsConsentService service = WsConsentSessionServiceFactory.getWsConsentService();
        PutPatientConsentResponse response = service.putPatientConsent(consentRequest);
        return response;
    }

    GetPatientConsentResponse retrievePatientConsentResponseWithSessionService() throws InstantiationException, ConnectorException {
        List<CDCONSENT> consentList = new ArrayList<CDCONSENT>();
        consentList.add(CDConsentBuilderUtil.createCDConsent("1.0", CDCONSENTvalues.RETROSPECTIVE));

        SelectGetPatientConsentType consentType = RequestObjectBuilderFactory.getConsentBuilder().createSelectGetPatientConsent(getPatient(), consentList);
        AuthorWithPatientAndPersonType author = getAuthor();

        GetPatientConsentRequest consentRequest = RequestObjectBuilderFactory.getRequestObjectBuilder().createGetRequest(author, consentType);

        TestUtils.logJaxbObject(consentRequest);
        // Service
        be.ehealth.businessconnector.wsconsent.session.WsConsentService service = WsConsentSessionServiceFactory.getWsConsentService();
        GetPatientConsentResponse response = service.getPatientConsent(consentRequest);

        TestUtils.logJaxbObject(response);
        return response;
    }

    RevokePatientConsentResponse revokePatientconsentWithSessionService(ConsentType consentType) throws InstantiationException, ConnectorException {

        consentType.setRevokedate(new DateTime());
        AuthorWithPatientAndPersonType author = getAuthor();

        RevokePatientConsentRequest consentRequest = RequestObjectBuilderFactory.getRequestObjectBuilder().createRevokeRequest(author, consentType);

        TestUtils.logJaxbObject(consentRequest);

        // Service
        be.ehealth.businessconnector.wsconsent.session.WsConsentService service = WsConsentSessionServiceFactory.getWsConsentService();
        RevokePatientConsentResponse response = service.revokePatientConsent(consentRequest);

        TestUtils.logJaxbObject(response);

        LOG.debug("Response: " + response);
        LOG.debug("isComplete =" + response.getAcknowledge().isIscomplete());
        List<ErrorType> errors = response.getAcknowledge().getErrors();
        for (ErrorType error : errors) {
            LOG.debug("URL : " + error.getUrl() + " errorMessage : " + error.getDescription().getValue());
        }
        return response;
    }

    void checkErrorMessage(RevokePatientConsentResponse revokePatientconsentResponse, String... errorMessages) {
        Assert.assertNotNull(revokePatientconsentResponse);
        Assert.assertNotNull(revokePatientconsentResponse.getAcknowledge());
        Assert.assertFalse(revokePatientconsentResponse.getAcknowledge().isIscomplete());
        Assert.assertTrue(revokePatientconsentResponse.getAcknowledge().getErrors().size() == errorMessages.length);
        for (int i = 0; i < errorMessages.length; i++) {
            String errorMessage = errorMessages[i];
            Assert.assertEquals(errorMessage, revokePatientconsentResponse.getAcknowledge().getErrors().get(i).getDescription().getValue());

        }
        LOG.info("checkErrorMessageContainsTheFollowingErrors for revokePatientconsentResponse: OK");
    }

    void checkIfSuccesFull(RevokePatientConsentResponse revokePatientconsentResponse) {
        Assert.assertNotNull(revokePatientconsentResponse);
        Assert.assertNotNull(revokePatientconsentResponse.getAcknowledge());
        Assert.assertTrue(revokePatientconsentResponse.getAcknowledge().isIscomplete());
        List<ErrorType> errors = revokePatientconsentResponse.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
        LOG.info("checkIfSuccesFull : revokePatientResponse indicates success");
    }

    void checkIfEqual(ConsentType expected, ConsentType actual) throws TechnicalConnectorException, SAXException, IOException {
        // name and firstname will not be returned !! -> this info is discarted in the backend
        // signDate : truncated to date level + different type in return type
        stripConsentType(expected);
        stripConsentType(actual);

        XmlAsserter.assertSimilar(expected, actual);
    }

    private void stripConsentType(ConsentType content) {
        content.getPatient().setFirstname(null);
        content.getPatient().setFamilyname(null);
        for (HcpartyType hcparty : content.getAuthor().getHcparties()) {
            List<IDHCPARTY> filteredList = new ArrayList<IDHCPARTY>();
            for (IDHCPARTY idhcparty : hcparty.getIds()) {
                if (idhcparty.getS() != IDHCPARTYschemes.INSS) {
                    filteredList.add(idhcparty);
                }
            }
            hcparty.getIds().clear();
            hcparty.getIds().addAll(filteredList);
        }
    }

    void checkErrorMessage(PutPatientConsentResponse secondPutConsentResponse, String... errorMessages) {
        Assert.assertNotNull(secondPutConsentResponse);
        Assert.assertNotNull(secondPutConsentResponse.getAcknowledge());
        Assert.assertFalse(secondPutConsentResponse.getAcknowledge().isIscomplete());
        Assert.assertTrue(secondPutConsentResponse.getAcknowledge().getErrors().size() == errorMessages.length);
        for (int i = 0; i < errorMessages.length; i++) {
            String errorMessage = errorMessages[i];
            Assert.assertEquals(errorMessage, secondPutConsentResponse.getAcknowledge().getErrors().get(i).getDescription().getValue());
        }
        LOG.info("checkErrorMessageContainsTheFollowingErrors : checked if response contained error messages : OK");
    }

    void checkIfSuccessFull(PutPatientConsentResponse putConsentResponse) {
        Assert.assertNotNull(putConsentResponse);
        Assert.assertNotNull(putConsentResponse.getAcknowledge());
        Assert.assertTrue(putConsentResponse.getAcknowledge().isIscomplete());
        List<ErrorType> errors = putConsentResponse.getAcknowledge().getErrors();
        Assert.assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
        LOG.info("checkIfSuccessFull : putConsentResponse was successFull");
    }


    void cleanConsents() throws TechnicalConnectorException, WsConsentBusinessConnectorException, InstantiationException {
        GetPatientConsentResponse retrievedPatientConsentResponse = retrievePatientConsentResponse();
        ConsentType consent = retrievedPatientConsentResponse.getConsent();
        if (consent != null) {
            LOG.info("cleanConsents : revoking existing consent");
            revokePatientconsent(consent);
        }
    }
}
