/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.attest.session;

import be.ehealth.business.mycarenetdomaincommons.domain.Ssin;
import be.ehealth.businessconnector.mycarenet.attest.builders.RequestObjectBuilder;
import be.ehealth.businessconnector.mycarenet.attest.builders.RequestObjectBuilderFactory;
import be.ehealth.businessconnector.mycarenet.attest.builders.ResponseObjectBuilder;
import be.ehealth.businessconnector.mycarenet.attest.builders.ResponseObjectBuilderFactory;
import be.ehealth.businessconnector.mycarenet.attest.domain.AttestBuilderRequest;
import be.ehealth.businessconnector.mycarenet.attest.domain.AttestBuilderResponse;
import be.ehealth.businessconnector.mycarenet.attest.domain.InputReference;
import be.ehealth.businessconnector.mycarenet.attest.session.AttestService;
import be.ehealth.businessconnector.mycarenet.attest.session.AttestSessionServiceFactory;
import be.ehealth.technicalconnector.adapter.XmlTimeAdapter;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.ehealth.technicalconnector.utils.DateUtils;
import be.ehealth.technicalconnector.utils.MarshallerHelper;
import be.ehealth.technicalconnector.utils.TemplateEngineUtils;
import be.fgov.ehealth.mycarenet.attest.protocol.v1.SendAttestationResponse;
import be.fgov.ehealth.standards.kmehr.schema.v1.FolderType;
import be.fgov.ehealth.standards.kmehr.schema.v1.Kmehrmessage;
import be.fgov.ehealth.standards.kmehr.schema.v1.TransactionType;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

/**
 * eAttest Integration Test. The scenarios numbering matches one specified by mycarenet.
 *
 * @author EHP
 */
@RunWith(Parameterized.class)
public class AttestIntegrationTest {

    @ClassRule
    public static SessionRule rule = SessionRule.withActiveSession().forEnvironment("acc").forProfession("persphysician").forUser("hannes").build();


    @Parameterized.Parameters(name = "{0},{1},{2}")
    public static Collection<Object[]> data() throws Exception {
        Collection<Object[]> testData = new ArrayList<Object[]>();
        testData.add(new Object[]{"100", "73052005540", "01", "11772830004"});
        testData.add(new Object[]{"600", "70021546287", "01", "10830643004"});
        testData.add(new Object[]{"100", "73052005540", "02", "11772830004"});
        testData.add(new Object[]{"600", "70021546287", "02", "10830643004"});
        testData.add(new Object[]{"100", "73052005540", "03", "11772830004"});
        testData.add(new Object[]{"600", "70021546287", "03", "10830643004"});
        testData.add(new Object[]{"100", "73052005540", "04", "11772830004"});
        testData.add(new Object[]{"600", "70021546287", "04", "10830643004"});
        testData.add(new Object[]{"100", "84101727579", "05", "11772830004"});
        testData.add(new Object[]{"600", "03051303986", "05", "10830643004"});
        testData.add(new Object[]{"500", "13070421120", "06", "14455473004"});
        testData.add(new Object[]{"600", "70021546287", "06", "10830643004"});
        testData.add(new Object[]{"100", "73052005540", "07", "11772830004"});
        testData.add(new Object[]{"600", "70021546287", "07", "10830643004"});
        testData.add(new Object[]{"100", "29041433972", "08", "11772830004"});
        testData.add(new Object[]{"300", "63042408660", "08", "19330813004"});
        testData.add(new Object[]{"600", "10090820056", "08", "10830643004"});
        testData.add(new Object[]{"100", "73052005540", "09", "11772830004"});
        testData.add(new Object[]{"600", "70021546287", "09", "10830643004"});
        testData.add(new Object[]{"600", "69021902691", "10", "10830643004"});
        testData.add(new Object[]{"300", "17031506487", "11", "19330813004"});
        testData.add(new Object[]{"600", "70021546287", "11", "10830643004"});
        testData.add(new Object[]{"300", "63042408660", "12", "19330813004"});
        testData.add(new Object[]{"600", "10090820056", "12", "10830643004"});
        testData.add(new Object[]{"300", "87052226861", "13", "19330813004"});
        testData.add(new Object[]{"600", "69021902691", "13", "10830643004"});
        testData.add(new Object[]{"300", "87120924439", "14", "19330813004"});
        testData.add(new Object[]{"600", "60042560332", "14", "10830643004"});
        testData.add(new Object[]{"300", "17031506487", "15", "19330813004"});
        testData.add(new Object[]{"600", "70021546287", "15", "10830643004"});
        testData.add(new Object[]{"300", "17031506487", "16", "19330813004"});
        testData.add(new Object[]{"600", "70021546287", "16", "10830643004"});
        testData.add(new Object[]{"300", "17031506487", "17", "19330813004"});
        testData.add(new Object[]{"600", "70021546287", "17", "10830643004"});
        testData.add(new Object[]{"300", "37061311820", "18", "19330813004"});
        testData.add(new Object[]{"600", "53081411750", "18", "10830643004"});

        return testData;
    }

    private String ssin;
    private String scenarioId;
    private String oa;
    private String dmgManager;



    public AttestIntegrationTest(String oa, String ssin, String scenarioId, String dmgManager) {
        super();
        this.ssin = ssin;
        this.scenarioId = scenarioId;
        this.oa = oa;
        this.dmgManager = dmgManager;

    }

    @Test
    public void sendAttestationTest() throws Exception {

        //given
        Ssin ssinValue = new Ssin(ssin);
        DateTime dateReference = new DateTime();
        RequestObjectBuilder builder = RequestObjectBuilderFactory.getRequestObjectBuilder();
        InputReference inputReference = new InputReference();
        Kmehrmessage kmehrRequest = buildKmehrmessage();
        AttestBuilderRequest attestBuilderRequest = builder.buildSendAttestationRequest(true, inputReference, ssinValue, dateReference, kmehrRequest);

        // The AttestBuilderRequest includes a before encryption SendTransactionRequest that can be used to apply xpath expressions
        // returned in the response in case of business error.
        ConnectorXmlUtils.dump(attestBuilderRequest.getSendTransactionRequest());
        
        AttestService service = AttestSessionServiceFactory.getAttestService();

        //when
        SendAttestationResponse response = service.sendAttestation(attestBuilderRequest.getSendAttestationRequest());
        ConnectorXmlUtils.dump(response);
        ResponseObjectBuilder responseBuilder = ResponseObjectBuilderFactory.getResponseObjectBuilder();
        AttestBuilderResponse builderResponse = responseBuilder.handleSendResponseType(response);

        //then
        assertTrue("Response id should have pattern oa number + '.' + 14 digits (we don't check that it is a date/time)", builderResponse.getSendTransactionResponse().getResponse().getId().getValue().matches("^"+oa+".(\\d{14}$)"));
        assertEquals("SignatureVerificationResult should contain no error", 0, builderResponse.getSignatureVerificationResult().getErrors().size());
        String expectedResponse = buildExpectedResponse(builderResponse, inputReference.getInputReference());
        XmlAsserter.assertSimilar(expectedResponse, ConnectorXmlUtils.toString(builderResponse.getSendTransactionResponse()));
    }

    private String buildExpectedResponse(AttestBuilderResponse builderResponse, String inputReference) throws Exception {
        Map<String, Object> velocityContext = new HashMap<String, Object>();
        velocityContext.put("today", DateTimeFormat.forPattern("yyyy-MM-dd").print(new DateTime()));
        velocityContext.put("responseId", builderResponse.getSendTransactionResponse().getResponse().getId().getValue());
        velocityContext.put("requestId", "18334780004" + "." + inputReference);
        velocityContext.put("responseTime", new XmlTimeAdapter().marshal(builderResponse.getSendTransactionResponse().getResponse().getTime()));
        velocityContext.put("requestTime",  new XmlTimeAdapter().marshal(builderResponse.getSendTransactionResponse().getResponse().getRequest().getTime()));

        if (builderResponse.getSendTransactionResponse().getKmehrmessage() != null) {
            velocityContext.put("headerTime", new XmlTimeAdapter().marshal(builderResponse.getSendTransactionResponse().getKmehrmessage().getHeader().getTime()));
            for (FolderType folder : builderResponse.getSendTransactionResponse().getKmehrmessage().getFolders()) {
                for (TransactionType transaction : folder.getTransactions()) {
                    velocityContext.put("transactionTime" + transaction.getIds().get(0).getValue(), new XmlTimeAdapter().marshal(transaction.getTime()));
                    if ("invoicingnumber".equals(transaction.getItem().get(0).getCds().get(0).getValue())) {
                        velocityContext.put("invoicingNumber", transaction.getItem().get(0).getContents().get(0).getTexts().get(0).getValue());
                    }
                }
            }
        }
        return TemplateEngineUtils.generate(velocityContext,"/scenarios/expected/responseDetail-" + oa + "-" + scenarioId + ".xml");
    }


    private Kmehrmessage buildKmehrmessage() {
        Map<String, Object> velocityContext = new HashMap<String, Object>();
        velocityContext.put("today", DateUtils.printDateTime(new DateTime()));
        velocityContext.put("ssin", ssin);
        velocityContext.put("DMGMANAGER_INA", dmgManager);
        String kmerh = TemplateEngineUtils.generate(velocityContext, "/scenarios/requests/TClearFile-AAA-" + scenarioId + ".xml");

        MarshallerHelper<Kmehrmessage, Kmehrmessage> kmehrRequestMarshaller = new MarshallerHelper<Kmehrmessage, Kmehrmessage>(Kmehrmessage.class, Kmehrmessage.class);
        return kmehrRequestMarshaller.toObject(kmerh);
    }

}
