/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.wsconsent.service;

import be.ehealth.business.kmehrcommons.CDConsentBuilderUtil;
import be.ehealth.business.kmehrcommons.helper.ErrorTypeManagement;
import be.ehealth.businessconnector.wsconsent.builders.RequestObjectBuilderFactory;
import be.ehealth.technicalconnector.exception.ConnectorException;
import be.fgov.ehealth.hubservices.core.v2.*;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDCONSENT;
import be.fgov.ehealth.standards.kmehr.cd.v1.CDCONSENTvalues;
import be.fgov.ehealth.standards.kmehr.id.v1.IDPATIENT;
import be.fgov.ehealth.standards.kmehr.id.v1.IDPATIENTschemes;
import be.fgov.ehealth.standards.kmehr.schema.v1.ErrorType;
import be.fgov.ehealth.technicalconnector.tests.utils.AssumeTools;

import org.joda.time.DateTime;
import org.junit.Before;
import org.junit.Test;

import java.util.ArrayList;
import java.util.List;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;


/**
 * Integration tests for the wsconsent connector
 */
public class WsConsentServiceUserIntegrationTest extends AbstractWsConsentServiceIntegrationTest {
    
    @Before
    public void checkIsEidEnabled() {
        AssumeTools.isEIDEnabled();
    }

    /**
     * Common scenario, declare a consent with a user by reading it's eid (you need to input
     * your eid with an eid reader, and you'll be the patient).
     */
    @Test
    public void testPutPatientConsent() throws Exception {
        ConsentType consentType = createNewConsentTypeForCurrentPatientAndAuthor();
        PutPatientConsentResponse response = putConsentForCurrentPatient(consentType);

        assertResponseIsValid(response);

        revokePatientconsent(consentType);
    }

    /**
     * Create a consent with a new born (age < 4 months), which doesn't require
     * an eid card number, just a inss.
     */
    @Test
    public void putConsentWithNewBorn() throws Exception {
        ConsentType consent = createConsentFor(newBorn());

        PutPatientConsentResponse response = putConsentForCurrentPatient(consent);

        assertResponseIsValid(response);

        revokePatientconsent(consent);
    }

    @Test
    public void testGetPatientConsent() throws Exception {
        ConsentType consentType = createNewConsentTypeForCurrentPatientAndAuthor();
        putConsentForCurrentPatient(consentType);

        GetPatientConsentResponse response = retrievePatientConsentResponse();
        assertNotNull(response);

        assertTrue(response.getAcknowledge().isIscomplete());
        List<ErrorType> errors = response.getAcknowledge().getErrors();
        assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
        LOG.debug("Response: " + response);

        revokePatientconsent(consentType);
    }

    @Test
    public void testRevokePatientConsent() throws Exception {
        ConsentType consentType = createNewConsentTypeForCurrentPatientAndAuthor();
        putConsentForCurrentPatient(consentType);

        RevokePatientConsentResponse response = revokePatientconsent(consentType);
        assertNotNull(response);
        assertNotNull(response.getAcknowledge());
        assertNotNull(response.getResponse());

        assertTrue(response.getAcknowledge().isIscomplete());
        List<ErrorType> errors = response.getAcknowledge().getErrors();
        assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

    @Test
    public void testPutPatientConsentWithSessionService() throws InstantiationException, ConnectorException {
        ConsentType consentType = createNewConsentTypeForCurrentPatientAndAuthor();
        PutPatientConsentResponse response = putConsentForCurrentPatientWithSessionService(consentType);
        assertNotNull(response);
        assertNotNull(response.getAcknowledge());
        assertNotNull(response.getResponse());

        assertTrue(response.getAcknowledge().isIscomplete());
        List<ErrorType> errors = response.getAcknowledge().getErrors();
        assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());

        revokePatientconsentWithSessionService(consentType);
    }

    @Test
    public void testGetPatientConsentWithSessionService() throws InstantiationException, ConnectorException {
        ConsentType consentType = createNewConsentTypeForCurrentPatientAndAuthor();
        putConsentForCurrentPatientWithSessionService(consentType);

        GetPatientConsentResponse response = retrievePatientConsentResponseWithSessionService();
        assertNotNull(response);

        assertTrue(response.getAcknowledge().isIscomplete());
        List<ErrorType> errors = response.getAcknowledge().getErrors();
        assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
        LOG.debug("Response: " + response);

        revokePatientconsentWithSessionService(consentType);
    }

    @Test
    public void testRevokePatientConsentWithSessionService() throws InstantiationException, ConnectorException {
        ConsentType consentType = createNewConsentTypeForCurrentPatientAndAuthor();
        putConsentForCurrentPatientWithSessionService(consentType);

        RevokePatientConsentResponse response = revokePatientconsentWithSessionService(consentType);
        assertNotNull(response);
        assertNotNull(response.getAcknowledge());
        assertNotNull(response.getResponse());

        assertTrue(response.getAcknowledge().isIscomplete());
        List<ErrorType> errors = response.getAcknowledge().getErrors();
        assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

    private static ConsentType createConsentFor(PatientIdType patient) throws Exception {
        List<CDCONSENT> consentList = new ArrayList<CDCONSENT>();
        consentList.add(CDConsentBuilderUtil.createCDConsent("1.0", CDCONSENTvalues.RETROSPECTIVE));
        return RequestObjectBuilderFactory.getConsentBuilder().createNewConsent(patient, consentList, new DateTime(), getAuthor());
    }

    private static PatientIdType newBorn() {
        PatientIdType patient = new PatientIdType();

        be.fgov.ehealth.standards.kmehr.id.v1.ObjectFactory objectFactoryIdPatient = new be.fgov.ehealth.standards.kmehr.id.v1.ObjectFactory();
        IDPATIENT idPatient = objectFactoryIdPatient.createIDPATIENT();
        idPatient.setSV("1.0");
        idPatient.setS(IDPATIENTschemes.INSS);
        idPatient.setValue("16101845918");

        patient.getIds().add(idPatient);

        return patient;
    }

    private static void assertResponseIsValid(PutPatientConsentResponse response) {
        assertNotNull(response);
        assertNotNull(response.getAcknowledge());
        assertNotNull(response.getResponse());

        List<ErrorType> errors = response.getAcknowledge().getErrors();
        assertTrue(ErrorTypeManagement.printErrors(errors), errors.isEmpty());
    }

}
