﻿/*
 * Copyright (c) eHealth
 */
using System;
using be.fgov.ehealth.errors.core.v1;
using be.fgov.ehealth.technicalconnector.tests.utils;
using ikvm.extensions;
using NUnit.Framework;
using java.math;
using be.ehealth.technicalconnector.exception;
using be.fgov.ehealth.technicalconnector.tests.session;
using be.ehealth.technicalconnector.utils;
using be.fgov.ehealth.samcivics.type.v1;

namespace be.ehealth.businessconnector.civics.session
{
	/**
	 * CIVICS Service Integration Tests This test shows the use of the CIVICS Service through the use of the Business Connector.
	 * 
	 * All Tests require the following pre-requisites to be met: - An active and valid eHealth-platform session (initiated via the Session
	 * Management component), in these tests this is done in the AbstractServiceTest class
	 */
	[TestFixture]
	public class CivicsSessionServiceIntegrationTest
	{
		[SetUp]
		public static void init() {
			SessionInitializer.init(".\\be.ehealth.businessconnector.civics.test.properties", SupportedLanguages.NET.getAbbreviation());
		}
		
		/**
		 * Version number of paragraphs to retrieve
		 */
		private static int VERSION = 1;
		
		/**
		 * Set this parameter to true if you want additional parameters in the request (additional parameter is relative to the version of the
		 * paragraph) : it can be a date (we will retrieve versions starting from this date) XOR the version of the paragraph to retrieve
		 */
		private static bool optional = false;
		
		/**
		 * Set this boolean to true if you want to select paragraphs with a version starting from this date If false, and optional set to true,
		 * the version VERSION will be selected
		 */
		private static bool date = false;
		
		
		/**
		 * Test the FindParagraphText operation from the CIVICS Service through the use of the Business Connector. This test retrieves the
		 * textual part and related data with regards to reimbursement stipulations.
		 * 
		 * @throws ConnectorException
		 */
		[Test]
		public void testFindParagraphText() {
			/*
			 * Set the values to use in this test
			 */
			// the chapter to search
			string chapterName = "IV";
			// the paragraph to search
			string paragraphName = "330100";
			// Set the language of the user
			LanguageType lang = LanguageType.NL;
			
			
			/*
			 * Create the request
			 */
			ParagraphAndVersesRequestType request = new ParagraphAndVersesRequestType();
			// set the chapter name
			request.setChapterName(chapterName);
			// set the paragraph name
			request.setParagraphName(paragraphName);
			// set the language
			request.setLanguage(lang);
			if (optional) {
				if (date) { // optionally: the start date, retrieve version starting from this date
					request.setStartDate(DateUtils.parseDate("2011-04-11"));
				} else {
					// optionally: the version of the paragraph to retrieve
					BigInteger optionalVersion = BigInteger.valueOf(VERSION);
					request.setVersion(optionalVersion);
				}
			}
			
			
			/*
			 * Invoke the business connector framework's eHealthBox's getMessageList operation
			 */
			CivicsService service = CivicsSessionServiceFactory.getCivicsService();
			FindParagraphTextResponse response = service.findParagraphText(request);
			
			/*
			 * Verify the response
			 */
			// Check if response has been received
			Assert.IsNotNull(response);
			// Check the chapter name
			Assert.IsNotNull(response.getParagraph().getChapterName());
		}
		
		/**
		 * Test the ParagraphIncludedSpecialties operation from the CIVICS Service through the use of the Business Connector. This test
		 * retrieves all specialities which are reimbursed within the requested paragraph.
		 * 
		 * @throws ConnectorException
		 */
		[Test]
		public  void getParagraphIncludedSpecialties() {
			/*
			 * Set the values to use in this test
			 */
			// the chapter to search
			String chapterName = "IV";
			// the paragraph to search
			String paragraphName = "330100";
			// Set the language of the user
			LanguageType lang = LanguageType.NL;
			
			/*
			 * Create the request
			 */
			ParagraphAndVersesRequestType request = new ParagraphAndVersesRequestType();
			// set the chapter name
			request.setChapterName(chapterName);
			// set the paragraph name
			request.setParagraphName(paragraphName);
			// set the language
			request.setLanguage(lang);
			if (optional) {
				if (date) { // optionally: the start date, retrieve version starting from this date
					request.setStartDate(DateUtils.parseDate("2011-04-11"));
				} else {
					// optionally: the version of the paragraph to retrieve
					BigInteger optionalVersion = BigInteger.valueOf(1);
					request.setVersion(optionalVersion);
				}
			}
			
			/*
			 * Invoke the business connector framework's eHealthBox's getMessageList operation
			 */
			CivicsService service = CivicsSessionServiceFactory.getCivicsService();
			GetParagraphIncludedSpecialitiesResponse response = service.getParagraphIncludedSpecialties(request);
			
			/*
			 * Verify the response
			 */
			// Check if response has been received
			Assert.IsNotNull(response);
		}
		
		/**
		 * Test the ParagraphExclusions operation from the CIVICS Service through the use of the Business Connector. This test retrieves the
		 * Specialities that may not be reimbursed in combination with other administered specialities for which there already is a refund
		 * agreement.
		 * 
		 * @throws ConnectorException
		 */
		[Test]
		public void testGetParagraphExclusions() {
			/*
			 * Set the values to use in this test
			 */
			// the chapter to search
			String chapterName = "IV";
			// the paragraph to search
			String paragraphName = "330100";
			// Set the language of the user
			LanguageType lang = LanguageType.NL;
			// optionally: the start date, retrieve version starting from this date
			
			/*
			 * Create the request
			 */
			ParagraphAndVersesRequestType request = new ParagraphAndVersesRequestType();
			// set the chapter name
			request.setChapterName(chapterName);
			// set the paragraph name
			request.setParagraphName(paragraphName);
			// set the language
			request.setLanguage(lang);
			if (optional) {
				if (date) { // optionally: the start date, retrieve version starting from this date
					request.setStartDate(DateUtils.parseDate("2011-04-11"));
				} else {
					// optionally: the version of the paragraph to retrieve
					BigInteger optionalVersion = BigInteger.valueOf(1);
					request.setVersion(optionalVersion);
				}
			}
			
			try {
				/*
				 * Invoke the business connector framework's eHealthBox's getMessageList operation
				 */
				CivicsService service = CivicsSessionServiceFactory.getCivicsService();
				GetParagraphExclusionsResponse response = service.getParagraphExclusions(request);
				
				/*
				 * Verify the response
				 */
				// Check if response has been received
				Assert.IsNotNull(response);
			} catch (SoaErrorException e) {
				if (!((LocalisedStringType)e.getErrorType().getMessages().get(0)).getValue().equalsIgnoreCase("No data found")) {
					throw e;
				};
			}
		}
		
		/**
		 * Test the ProfessionalAuthorizations operation from the CIVICS Service through the use of the Business Connector. This test retrieves
		 * the qualification list, stipulating requirements concerning the professional qualification.
		 * 
		 * @throws ConnectorException
		 */
		[Test]
		public void getProfessionalAuthorizations() {
			/*
			 * Set the values to use in this test
			 */
			// the paragraph to search
			string qualificationlist = "4";
			// Set the language of the user
			LanguageType lang = LanguageType.NL;
			// the input date
			// Calendar inputDate = Calendar.getInstance();
			// inputDate.set(2010, 5, 11);
			
			/*
			 * Create the request
			 */
			GetProfessionalAuthorisationsRequest request = new GetProfessionalAuthorisationsRequest();
			// set the input date
			// request.setInputDate(inputDate);
			// set the language
			request.setLanguage(lang);
			// set the qualification list
			request.setQualificationList(qualificationlist);
			
			/*
			 * Invoke the business connector framework's eHealthBox's getMessageList operation
			 */
			CivicsService service = CivicsSessionServiceFactory.getCivicsService();
			GetProfessionalAuthorisationsResponse response = service.getProfessionalAuthorizations(request);
			
			/*
			 * Verify the response
			 */
			// Check if response has been received
			Assert.IsNotNull(response);
		}
		
		/**
		 * Test the GetAddedDocuments operation from the CIVICS Service through the use of the Business Connector. This test retrieves the
		 * description and identification of the structured forms annexed to a paragraph.
		 * 
		 * @throws ConnectorException
		 */
		[Test]
		public void testGetAddedDocuments() {
			/*
			 * Set the values to use in this test
			 */
			// the chapter to search
			string chapterName = "IV";
			// the paragraph to search
			string paragraphName = "1070000";
			// Set the language of the user
			LanguageType lang = LanguageType.NL;
			
			
			/*
			 * Create the request
			 */
			ParagraphAndVersesRequestType request = new ParagraphAndVersesRequestType();
			// set the chapter name
			request.setChapterName(chapterName);
			// set the paragraph name
			request.setParagraphName(paragraphName);
			// set the language
			request.setLanguage(lang);
			if (optional) {
				if (date) { // optionally: the start date, retrieve version starting from this date
					request.setStartDate(DateUtils.parseDate("2011-04-11"));
				} else {
					// optionally: the version of the paragraph to retrieve
					BigInteger optionalVersion = BigInteger.valueOf(1);
					request.setVersion(optionalVersion);
				}
			}
			
			/*
			 * Invoke the business connector framework's eHealthBox's getMessageList operation
			 */
			CivicsService service = CivicsSessionServiceFactory.getCivicsService();
			try {
				GetAddedDocumentsResponse response = service.getAddedDocuments(request);
				
				/*
				 * Verify the response
				 */
				// Check if response has been received
				Assert.IsNotNull(response);
				
				
			} catch (SoaErrorException e) {
				if (!((LocalisedStringType)e.getErrorType().getMessages().get(0)).getValue().equalsIgnoreCase("No data found")) {
					throw e;
				};
			}
			
			
		}
		
		/**
		 * Test the FindReimbursementConditions operation from the CIVICS Service through the use of the Business Connector. This test retrieves
		 * all information on prices and reimbursement amounts.
		 * 
		 * @throws ConnectorException
		 */
		[Test]
		public void testFindReimbursementConditions() {
			/*
			 * Set the values to use in this test
			 */
			// the chapter to search
			string chapterName = "IV";
			// the paragraph to search
			string paragraphName = "330100";
			// Set the language of the user
			LanguageType lang = LanguageType.NL;
			
			/*
			 * Create the request
			 */
			FindReimbursementConditionsRequest request = new FindReimbursementConditionsRequest();
			// set the chapter name
			request.setChapterName(chapterName);
			// set the paragraph name
			request.setParagraphName(paragraphName);
			// set the language
			request.setLanguage(lang);
			if (optional) {
				if (date) { // optionally: the start date, retrieve version starting from this date
					request.setStartDate(DateUtils.parseDate("2011-04-11"));
				} else {
					// optionally: the version of the paragraph to retrieve
					BigInteger optionalVersion = BigInteger.valueOf(1);
					request.setVersion(optionalVersion);
				}
			}
			
			/*
			 * Invoke the business connector framework's eHealthBox's getMessageList operation
			 */
			CivicsService service = CivicsSessionServiceFactory.getCivicsService();
			FindReimbursementConditionsResponse response = service.findReimbursementConditions(request);
			
			/*
			 * Verify the response
			 */
			// Check if response has been received
			Assert.IsNotNull(response);
		}
		
		[Test]
		public void testFindCNK() {
			string productName = "PLAVIX";
			FindCNKRequest request = new FindCNKRequest();
			request.setProductName(productName);
			request.setLanguage(LanguageType.FR);
			
			CivicsService service = CivicsSessionServiceFactory.getCivicsService();
			FindCNKResponse response = service.findCNK(request);
			
			Assert.IsNotNull(response);
			
			
		}
	}
}
