/*
 * Copyright (c) eHealth
 */

package be.fgov.ehealth.technicalconnector.ra.domain;

import be.ehealth.technicalconnector.beid.BeIDCardFactory;
import be.ehealth.technicalconnector.config.ConfigFactory;
import be.ehealth.technicalconnector.session.Session;
import be.ehealth.technicalconnector.utils.ConnectorIOUtils;
import be.fgov.ehealth.technicalconnector.ra.enumaration.Language;
import be.fgov.ehealth.technicalconnector.ra.enumaration.UsageType;
import be.fgov.ehealth.technicalconnector.tests.beid.DummyBeIDAdaptor;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.BeIDInfoRule;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.session.SessionInitializer;
import be.fgov.ehealth.technicalconnector.tests.utils.AssumeTools;
import org.apache.commons.lang.StringUtils;
import org.junit.*;

import java.awt.*;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.cert.X509Certificate;

import static be.fgov.ehealth.technicalconnector.ra.domain.ContractIntegrationTest.generateCert;
import static be.fgov.ehealth.technicalconnector.tests.session.SessionInitializer.getSessionProperty;

/**
 * @author EHP
 */
public class NewCertificateContractTest {


    private String phonePrivate;
    private String mailPrivate;
    private String mailGeneral;
    private String phoneGeneral;
    private X509Certificate orgCert;
    private X509Certificate persCert;

    @Rule
    public BeIDInfoRule beid = BeIDInfoRule.activateEid().withName().Alice();

    @ClassRule
    public static SessionRule session = SessionRule.withInactiveSession().build();

    @Before
    public void init() throws Exception {
        phonePrivate = getSessionProperty("test.ra.private.phone");
        mailPrivate = getSessionProperty("test.ra.private.mail");
        mailGeneral = getSessionProperty("test.ra.general.mail");
        phoneGeneral = getSessionProperty("test.ra.general.phone");
        orgCert = generateCert(" O=Federal Government,OU=eHealth-platform Belgium,OU=APOTHEEL ACCOU NV,OU=NIHII-PHARMACY\\=34240406,C=BE,CN=NIHII-PHARMACY\\=34240406");
        persCert = generateCert("O=Federal Government,OU=eHealth-platform Belgium,OU=TEST PERSON,OU=SSIN\\=01020300156,C=BE,CN=SSIN\\=01020300156");
    }

    @Test
    public void newCertificateContractInDutchWithPrivateContactDataWithEidThroughConstructor() throws Exception {
        newCertificateContract(new ContactData(phonePrivate, mailPrivate, Language.NL), new DistinguishedName());
    }

    @Test
    public void newCertificateContractInDutchWithPrivateContactDataWithoutEidThroughConstructor() throws Exception {
        newCertificateContract(new ContactData(phonePrivate, mailPrivate, Language.NL), new DistinguishedName(persCert.getSubjectX500Principal()));
    }

    @Test
    public void newCertificateContractInFrenchWithPrivateContactDataWithEidThroughConstructor() throws Exception {
        newCertificateContract(new ContactData(phonePrivate, mailPrivate, Language.FR), new DistinguishedName());
    }

    @Test
    public void newCertificateContractInFrenchWithPrivateContactDataWithoutEidThroughConstructor() throws Exception {
        newCertificateContract(new ContactData(phonePrivate, mailPrivate, Language.FR), new DistinguishedName(persCert.getSubjectX500Principal()));
    }

    @Test
    public void newCertificateContractInFrenchWithPrivateAndGeneralContactDataWithEidThroughConstructor() throws Exception {
        newCertificateContract(new ContactData(mailGeneral, phoneGeneral, mailGeneral, mailPrivate, Language.FR), new DistinguishedName());
    }

    @Test
    public void newCertificateContractInDutchWithPrivateAndGeneralContactDataWithEidThroughConstructor() throws Exception {
        newCertificateContract(new ContactData(mailGeneral, phoneGeneral, mailGeneral, mailPrivate, Language.NL), new DistinguishedName());
    }

    @Test
    public void newCertificateContractInFrenchWithPrivateAndGeneralContactDataWithoutEidThroughConstructor() throws Exception {
        newCertificateContract(new ContactData(mailGeneral, phoneGeneral, mailGeneral, mailPrivate, Language.FR), new DistinguishedName(persCert.getSubjectX500Principal()));
    }

    private static NewCertificateContract newCertificateContract(ContactData contact, DistinguishedName name) throws Exception {
        NewCertificateContract contract = new NewCertificateContract(name, contact, UsageType.CODAGE);
        view(contract);
        return contract;
    }

    private static void view(Contract contract) throws IOException {
        Assert.assertFalse(StringUtils.contains(contract.getContent(), "$"));
        if (!GraphicsEnvironment.isHeadless()) {
            AssumeTools.isUserInteractionEnabled();
            File htmlFile = File.createTempFile("contract", ".html");
            FileOutputStream fos = new FileOutputStream(htmlFile);
            try {
                fos.write(contract.getContract().getBytes());
            } finally {
                ConnectorIOUtils.closeQuietly(fos);
            }
            Desktop.getDesktop().browse(htmlFile.toURI());
        }
    }
}