/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.chapterIV.builders;

import java.io.StringWriter;

import javax.xml.bind.JAXB;

import org.junit.Assert;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import be.ehealth.businessconnector.chapterIV.domain.ChapterIVReferences;
import be.ehealth.businessconnector.chapterIV.exception.ChapterIVBusinessConnectorException;
import be.ehealth.businessconnector.chapterIV.testutil.TestUtils;
import be.ehealth.businessconnector.chapterIV.utils.FolderTypeUtils;
import be.fgov.ehealth.chap4.protocol.v1.ConsultChap4MedicalAdvisorAgreementRequest;
import be.fgov.ehealth.standards.kmehr.id.v1.IDPATIENT;
import be.fgov.ehealth.standards.kmehr.schema.v1.FolderType;
import be.fgov.ehealth.technicalconnector.tests.utils.TestPropertiesLoader;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;


/**
 * Unit testing the ConsultationBuilder
 * 
 * @author EHP
 * 
 */
public class ConsultationBuilderTest extends AbstractBuilderTest {

    @BeforeClass
    public static void init() throws Exception {
        // only integration tests need to run for all professionTypes
        Assume.assumeTrue("persphysician".equals(TestPropertiesLoader.getProperties("/be.ehealth.businessconnector.chapterIV.test.properties").getProperty("session.professionType")));
    }


    private static final Logger LOG = LoggerFactory.getLogger(ConsultationBuilderTest.class);

    /**
     * Test with an invalid kmehrmessageType
     * 
     * @throws Exception
     */
    @Test(expected = ChapterIVBusinessConnectorException.class)
    public void testInvalidRequest() throws Exception {

        ConsultationBuilder builder = getFactory().getConsultationBuilder();
        FolderType folder = null;
        boolean isTest = true;
        String commonReference = "33";
        ChapterIVReferences references = new ChapterIVReferences(commonReference);

        builder.buildRequest(folder, isTest, references, FolderTypeUtils.retrieveConsultationStartDateOrAgreementStartDate(folder));


    }


    /**
     * Test with a valid kmehrmessageType and with a ssin.
     * 
     * @throws Exception
     */

    @Test
    public void testRequestWithoutSsin() throws Exception {

        ConsultationBuilder builder = getFactory().getConsultationBuilder();

        FolderType folder = JAXB.unmarshal(ConsultationBuilderTest.class.getResourceAsStream("/examples/sample-chapter4-consultation-request-without-ssin.xml"), FolderType.class);
        boolean isTest = true;
        String commonReference = "33";
        ChapterIVReferences references = new ChapterIVReferences(commonReference);


        ConsultChap4MedicalAdvisorAgreementRequest request = builder.buildRequest(folder, isTest, references, FolderTypeUtils.retrieveConsultationStartDateOrAgreementStartDate(folder));
        StringWriter sw = new StringWriter();
        JAXB.marshal(request, sw);
        LOG.debug(sw.toString());

        Assert.assertNotNull(request.getRecordCommonInput().getInputReference().toString());
        Assert.assertEquals(commonReference, request.getCommonInput().getInputReference().toString());
        Assert.assertNull("commonNIPReference should not be filled out", request.getCommonInput().getNIPReference());

        Assert.assertEquals(props.getProperty("test.hcp.inss"), request.getCommonInput().getOrigin().getCareProvider().getPhysicalPerson().getSsin().getValue());
        Assert.assertEquals(props.getProperty("test.hcp.nihii"), request.getCommonInput().getOrigin().getCareProvider().getNihii().getValue().getValue());
        XmlAsserter.assertSimilar(TestUtils.getFileContents("/expected/builders/resultConsultRequestWithoutSsin.xml"), sw.toString());

    }

    /**
     * Test with a valid kmehrmessageType and without a SSIN
     * 
     * @throws Exception
     */
    @Test
    public void testRequestWithSsin() throws Exception {

        ConsultationBuilder builder = getFactory().getConsultationBuilder();

        FolderType folder = JAXB.unmarshal(ConsultationBuilderTest.class.getResourceAsStream("/examples/sample-chapter4-consultation-request-with-ssin.oa500.xml"), FolderType.class);
        for (IDPATIENT idPatient : folder.getPatient().getIds()) {
            idPatient.setValue("79031425139");
        }
        boolean isTest = true;
        String commonReference = "33";
        ChapterIVReferences references = new ChapterIVReferences(commonReference);


        ConsultChap4MedicalAdvisorAgreementRequest request = builder.buildRequest(folder, isTest, references, FolderTypeUtils.retrieveConsultationStartDateOrAgreementStartDate(folder));
        StringWriter sw = new StringWriter();
        JAXB.marshal(request, sw);
        LOG.debug(sw.toString());

        Assert.assertNotNull(request.getRecordCommonInput().getInputReference().toString());
        Assert.assertEquals(commonReference, request.getCommonInput().getInputReference().toString());
        Assert.assertNull("commonNIPReference should not be filled out", request.getCommonInput().getNIPReference());
        Assert.assertEquals(props.getProperty("test.hcp.inss"), request.getCommonInput().getOrigin().getCareProvider().getPhysicalPerson().getSsin().getValue());
        Assert.assertEquals(props.getProperty("test.hcp.nihii"), request.getCommonInput().getOrigin().getCareProvider().getNihii().getValue().getValue());
        XmlAsserter.assertSimilar(TestUtils.getFileContents("/expected/builders/resultConsultRequestWithSsin.xml"), sw.toString());

    }


}
